// Copyright (C) Mikko Apo (apo@iki.fi)
// The following code may be used to write free software
// if credit is given to the original author.
// Using it for anything else is not allowed without permission
// from the author.

/*

  Revision history:

  1.02	Got a little guilty consciousness because of not obeying
    the WM_READ mode. <thru> now bypasses machine.
  1.01	Fixed zero crossings miscalculation and added some
	more info. Thanks to Tamas for reporting.
  1.0	First release

*/

#include <string.h>
#include <math.h>
#include "../mdk.h"

#define COMMAND_STRING "Show analysis...\nReset analysis\nAbout..."
#define MACHINE_NAME "cheapo statistics"
#define SHORT_NAME "ch.stats"
#define MACHINE_AUTHOR "Mikko Apo (apo@iki.fi)"
#define MAX_TRACKS		0
#define MIN_TRACKS		0
#define NUMGLOBALPARAMETERS 0
#define NUMTRACKPARAMETERS 0
#define NUMATTRIBUTES 0
#define __VERSION__ "1.02"

// Machine's info

CMachineInfo const MacInfo = 
{
	MT_EFFECT,MI_VERSION,MIF_DOES_INPUT_MIXING,MIN_TRACKS,MAX_TRACKS,
	NUMGLOBALPARAMETERS,NUMTRACKPARAMETERS,NULL,NUMATTRIBUTES,NULL,
#ifdef _DEBUG
	MACHINE_NAME" [DEBUG]"
#else
	MACHINE_NAME
#endif
	,SHORT_NAME,MACHINE_AUTHOR,COMMAND_STRING
};

class miex : public CMDKMachineInterfaceEx
{

};

class mi : public CMDKMachineInterface
{
public:
	mi();

	virtual void Command(int const i);

	virtual void MDKInit(CMachineDataInput * const pi);
	virtual bool MDKWork(float *psamples, int numsamples, int const mode);
	virtual bool MDKWorkStereo(float *psamples, int numsamples, int const mode);
	virtual void MDKSave(CMachineDataOutput * const po) { }

	public:
	virtual CMDKMachineInterfaceEx *GetEx() { return &ex; }
	virtual void OutputModeChanged(bool stereo);

	public:
	miex ex;

private:
	void print_amp(char *txt, float num);
	void print_ana1(char *txt, float max,float min);
	void print_suggest(char *txt, float max,float min);
    void print_time(char *txt,unsigned long samples);
    void print_channel(char *txt,float maxlevelfound, float minlevelfound, unsigned long time_maxvalue, unsigned long time_minvalue, double dcsum,double rms,unsigned long zero_cross);

	float left_maxlevelfound,left_minlevelfound;
	float right_maxlevelfound,right_minlevelfound;
	double left_dcsum,right_dcsum;
	double left_rms,right_rms;
	float prev_left,prev_right;
	unsigned long left_zerocross, right_zerocross;
	unsigned long dcnum;
	unsigned long time_left_minvalue,time_left_maxvalue,time_right_minvalue,time_right_maxvalue;
	bool stereo_mode;
};


DLL_EXPORTS

mi::mi()
{
	GlobalVals = NULL;
	TrackVals = NULL;
	AttrVals = NULL;

}

// Produces output for analysis

void mi::print_amp(char *txt, float num)
{
	sprintf(txt,"%ssample:%+.1f (",txt,num);
	if(num)
	{
	  sprintf(txt,"%s%.1fdB; %+.2f%%)",txt,(float)(20.0*log10(fabs(num)/( (num>0)?32767.0:32768.0) )),(float)((100*num)/( (num>0)?32767.0:32768.0)));
	} else
	{
	  sprintf(txt,"%s-inf dB; 0.00%%)",txt);
	}

	if(num>32767.0||num<-32768.0)
	{
		sprintf(txt,"%s *** Possible clipping ***",txt);
	}
	sprintf(txt,"%s\n",txt);
}

void mi::print_suggest(char *txt, float max,float min)
{
	float tmp_max,tmp_maxrange,tmp;
	if(max>32767.0||min<-32768.0)
	{
		if(fabs(max)<fabs(min))
		{
			tmp_max=(float)fabs(min);
			tmp_maxrange=32768.0;
		} else
		{
			tmp_max=(float)fabs(max);
			tmp_maxrange=32767.0;
		}
		sprintf(txt,"%sThe analysed signal exceeds the normal range between -32768.0 and 32767.0\nIt might clip in the following effect and it will clip if connected to master output.\n",txt);
		tmp=(float)16384.0*(tmp_maxrange/tmp_max);
		sprintf(txt,"%sThe input volume slider of the next machine should be set to under %d [%.2f%%].\n",txt,(int)tmp,(100*tmp)/16384 );
		sprintf(txt,"%sThe master output volume slider should have a value over ",txt);
		tmp=(float)(log10(tmp_maxrange/tmp_max)*((20.0*16384.0)/(-80.0)));
		sprintf(txt,"%s%d [%.2f%%] to prevent clipping.\n",txt,(int)tmp,(100*tmp)/16384);
	} else
	{
		sprintf(txt,"%sThe analysed signal was in the normal range between -32768.0 and 32767.0. \nThere is no need to adjust the volume sliders.\n",txt);
	}
}

void mi::print_time(char *txt,unsigned long samples)
{
  unsigned long secs;
  unsigned long samplespersec=pMasterInfo->SamplesPerSec;
  unsigned long samplespertick=pMasterInfo->SamplesPerTick;
  secs=(unsigned int)(samples/samplespersec);
  sprintf(txt,"%s %ld ticks [time: %d:%02d:%02d.%03d (%ld samples)]\n",txt,samples/samplespertick,secs/3600,(secs%3600)/60,secs%60,(1000*(samples%(samplespersec)))/samplespersec,samples);
}

void mi::print_channel(char *txt,float maxlevelfound, float minlevelfound, unsigned long time_maxvalue, unsigned long time_minvalue, double dcsum,double rms,unsigned long zero_cross)
{
	sprintf(txt,"%sMax ",txt);
	print_amp(txt,maxlevelfound);
    sprintf(txt,"%sMax value location: at",txt);
    print_time(txt,time_maxvalue);

	sprintf(txt,"%sMin ",txt);
	print_amp(txt,minlevelfound);
    sprintf(txt,"%sMin value location: at",txt);
    print_time(txt,time_minvalue);

	sprintf(txt,"%sDC Offset (average): %.1f (",txt,dcsum/dcnum);
	if(dcsum/dcnum)
	{
	  sprintf(txt,"%s%.1fdB; %+.2f%%)\n",txt,(float)(20.0*log10(fabs(dcsum/(dcnum*32768.0)))),(float)((100*(dcsum/dcnum))/32768.0));
	} else
	{
	  sprintf(txt,"%s-inf dB; 0.00%%)\n",txt);
	}
	rms=sqrt(rms/dcnum);
	sprintf(txt,"%sRMS Power: %.1f (",txt,rms);
	if(rms)
	{
	  sprintf(txt,"%s%.1fdB; %+.2f%%)\n",txt,(float)(20.0*log10(rms/32768.0)),(float)((100*rms)/32768.0));
	} else
	{
	  sprintf(txt,"%s-inf dB; 0.00%%)\n",txt);
	}
	sprintf(txt,"%sZero crossings: %ld (%.2fHz)\n",txt,zero_cross,(float)(zero_cross*pMasterInfo->SamplesPerSec)/(dcnum*2));
}

void mi::Command(int const i)
{
	static char txt[2000];
	switch(i)
	{
	case 0:
		if(dcnum)
		{
		  sprintf(txt,"Sound analysis:\n\nStatistical data collected:");
		  print_time(txt,dcnum);
		  if(stereo_mode)
		  {
			  sprintf(txt,"%s\nLeft channel:\n\n",txt);
			  print_channel(txt,left_maxlevelfound,left_minlevelfound,time_left_maxvalue, time_left_minvalue, left_dcsum,left_rms,left_zerocross);
			  sprintf(txt,"%s\nRight channel:\n\n",txt);
			  print_channel(txt,right_maxlevelfound,right_minlevelfound,time_right_maxvalue, time_right_minvalue, right_dcsum,right_rms,right_zerocross);
			  sprintf(txt,"%s\nVolume slider suggestion based on max and min levels of both channels:\n\n",txt);
			  print_suggest(txt,(left_maxlevelfound>right_maxlevelfound)?left_maxlevelfound:right_maxlevelfound,(left_minlevelfound<right_minlevelfound)?left_minlevelfound:right_minlevelfound);
		  } else
		  {
			  sprintf(txt,"%s\nMono signal:\n\n",txt);
			  print_channel(txt,left_maxlevelfound,left_minlevelfound,time_left_maxvalue, time_left_minvalue, left_dcsum,left_rms,left_zerocross);
			  sprintf(txt,"%s\nVolume slider suggestion based on max and min levels:\n\n",txt);
			  print_suggest(txt,left_maxlevelfound,left_minlevelfound);
		  }
		} else
		{
			sprintf(txt,"*** No sound analysed ***\n");
		}
		pCB->MessageBox(txt);
		break;
	case 1:
		left_maxlevelfound=right_maxlevelfound=-9999999.0;
		left_minlevelfound=right_minlevelfound=9999999.0;
		left_dcsum=right_dcsum=0.0;
		left_rms=right_rms=0.0;
		dcnum=0;
		time_left_minvalue=time_left_maxvalue=time_right_minvalue=time_right_maxvalue=0;
		prev_left=prev_right=0.0;
		left_zerocross=right_zerocross=0;
		break;
	case 2:
		pCB->MessageBox(MACHINE_NAME"\n\nBuild date: "__DATE__"\nVersion: "__VERSION__"\nCoded by: "MACHINE_AUTHOR"\nThanks to Oskari and other #buzz developers.\n\nCheck out http://www.iki.fi/apo/buzz/\nfor more buzz stuff.\n\nExcellent skin made by Hymax.");
		break;
	}
}

void mi::MDKInit(CMachineDataInput * const pi)
{
	stereo_mode=false;
	Command(1);
}

void mi::OutputModeChanged(bool stereo)
{
	stereo_mode=stereo;
	Command(1);
}

bool mi::MDKWorkStereo(float *psamples, int numsamples, int const mode)
{
	float l,r;
	if ((mode==WM_WRITE)||(mode==WM_NOIO))
	{
		return false;
	}

	if (mode == WM_READ)		// <thru>
		return true;

	do 
	{
		l=psamples[0];
		r=psamples[1];
		psamples+=2;
		left_dcsum+=l;
		left_rms+=l*l;
		if(l>left_maxlevelfound)
		{
			left_maxlevelfound=l;
			time_left_maxvalue=dcnum;
		}
		if(l<left_minlevelfound)
		{
			left_minlevelfound=l;
			time_left_minvalue=dcnum;
		}
		if(prev_left>=0)
		{
			if(l<0)
			left_zerocross++;
		} else
		{
			if(l>=0)
			left_zerocross++;
		}
		prev_left=l;

		right_dcsum+=r;
		right_rms+=r*r;
		if(r>right_maxlevelfound)
		{
			right_maxlevelfound=r;
			time_right_maxvalue=dcnum;
		}
		if(r<right_minlevelfound)
		{
			right_minlevelfound=r;
			time_right_minvalue=dcnum;
		}
		if(prev_right>=0)
		{
			if(r<0)
			right_zerocross++;
		} else
		{
			if(r>=0)
			right_zerocross++;
		}
		prev_right=r;
		dcnum++;
	} while(--numsamples);
	return true;
}

bool mi::MDKWork(float *psamples, int numsamples, int const mode)
{
	float l;
	if ((mode==WM_WRITE)||(mode==WM_NOIO))
	{
		return false;
	}

	if (mode == WM_READ)		// <thru>
		return true;

	do 
	{
		l=*psamples;
		psamples++;
		left_dcsum+=l;
		left_rms+=l*l;
		if(l>left_maxlevelfound)
		{
			left_maxlevelfound=l;
			time_left_maxvalue=dcnum;
		}
		if(l<left_minlevelfound)
		{
			left_minlevelfound=l;
			time_left_minvalue=dcnum;
		}
		if(prev_left>=0)
		{
			if(l<0)
			left_zerocross++;
		} else
		{
			if(l>=0)
			left_zerocross++;
		}
		prev_left=l;
		dcnum++;
	} while(--numsamples);
	return true;
}
